import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Scissors, Ruler, Palette, Sparkles, Calendar, CheckCircle, Award, Users, Clock, Heart, Star, Quote, BookOpen, GraduationCap, Zap } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedLevel, setSelectedLevel] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const levels = [
    { 
      id: 'basic', 
      name: 'Базовый', 
      icon: Scissors,
      color: '#E8B4B8',
      description: 'Основы шитья для начинающих. Научитесь работать с тканью и швейной машинкой.',
      duration: '6 недель',
      students: '800+',
      rating: '4.9'
    },
    { 
      id: 'cutting', 
      name: 'Кройка', 
      icon: Ruler,
      color: '#C4B5D9',
      description: 'Изучите техники раскроя ткани и создание выкроек для различных моделей.',
      duration: '8 недель',
      students: '650+',
      rating: '4.8'
    },
    { 
      id: 'modeling', 
      name: 'Моделирование', 
      icon: Palette,
      color: '#A8C4A2',
      description: 'Создание собственных моделей одежды и работа с формой и силуэтом.',
      duration: '10 недель',
      students: '500+',
      rating: '4.9'
    },
    { 
      id: 'author', 
      name: 'Авторская одежда', 
      icon: Sparkles,
      color: '#D4AF37',
      description: 'Разработка уникальных коллекций и создание авторских моделей одежды.',
      duration: '12 недель',
      students: '350+',
      rating: '5.0'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedLevel || !startDate) {
      alert('Пожалуйста, выберите уровень и дату начала')
      return
    }

    const saved = localStorage.getItem('sewingCourses')
    const courses = saved ? JSON.parse(saved) : []
    
    const levelData = levels.find(l => l.id === selectedLevel)
    
    // Находим максимальный ID и добавляем 1
    const maxId = courses.length > 0 ? Math.max(...courses.map(c => c.id)) : 0
    const newId = maxId + 1
    
    const newCourse = {
      id: newId,
      project: `Проект ${levelData.name}`,
      teacher: 'Преподаватель будет назначен',
      completionDate: startDate,
      status: 'active',
      resultPhoto: null
    }
    
    courses.push(newCourse)
    localStorage.setItem('sewingCourses', JSON.stringify(courses))
    
    setIsSubmitted(true)
    
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  return (
    <div className="enroll-page-sewing">
      <div className="container">
        <section className="enroll-header-sewing">
          <div className="header-badge-sewing">
            <Scissors size={20} />
            <span>ЗАПИСЬ НА КУРС</span>
          </div>
          <h1 className="enroll-title-sewing">
            Записаться на <span className="title-accent-sewing">курс</span>
          </h1>
          <p className="enroll-description-sewing">
            Выберите уровень обучения и дату начала. Все курсы включают практические занятия и обратную связь от мастеров.
          </p>
        </section>

        <section className="enroll-form-section-sewing">
          <form className="form-sewing" onSubmit={handleSubmit}>
            <div className="levels-header-sewing">
              <h2 className="levels-title-sewing">Выберите уровень обучения</h2>
              <p className="levels-description-sewing">
                Каждый уровень рассчитан на определенный опыт и навыки
              </p>
            </div>
            <div className="levels-selection-sewing">
              {levels.map((level) => {
                const Icon = level.icon
                return (
                  <button
                    key={level.id}
                    type="button"
                    className={`level-card-sewing ${selectedLevel === level.id ? 'selected' : ''}`}
                    onClick={() => setSelectedLevel(level.id)}
                    style={{ '--level-color': level.color }}
                  >
                    <div className="level-icon-wrapper-sewing">
                      <Icon size={40} />
                    </div>
                    <h3 className="level-name-sewing">{level.name}</h3>
                    <p className="level-desc-sewing">{level.description}</p>
                    <div className="level-stats-sewing">
                      <div className="level-stat-sewing">
                        <span className="stat-label-sewing">Длительность:</span>
                        <span className="stat-value-sewing">{level.duration}</span>
                      </div>
                      <div className="level-stat-sewing">
                        <span className="stat-label-sewing">Студентов:</span>
                        <span className="stat-value-sewing">{level.students}</span>
                      </div>
                      <div className="level-stat-sewing">
                        <span className="stat-label-sewing">Рейтинг:</span>
                        <span className="stat-value-sewing">{level.rating}</span>
                      </div>
                    </div>
                    {selectedLevel === level.id && (
                      <div className="check-badge-sewing">
                        <CheckCircle size={24} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedLevel && (
              <div className="date-section-sewing">
                <div className="date-label-sewing">
                  <Calendar size={20} />
                  <span>Дата начала курса</span>
                </div>
                <input
                  type="date"
                  className="date-input-sewing"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDateStr}
                  max={maxDateStr}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-sewing" 
              disabled={!selectedLevel || !startDate || isSubmitted}
            >
              <span>Записаться на курс</span>
            </button>

            {isSubmitted && (
              <div className="success-message-sewing">
                <CheckCircle size={32} />
                <span>Запись успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <SuccessStoriesSection />

        <BenefitsSection />
      </div>
    </div>
  )
}

const SuccessStoriesSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const stories = [
    {
      name: 'Анна К.',
      achievement: 'Создала коллекцию из 10 платьев',
      text: 'За полгода обучения я создала свою первую коллекцию платьев! Преподаватели настоящие профессионалы.',
      image: 'linear-gradient(135deg, #E8B4B8 0%, #C4B5D9 100%)',
      rating: 5
    },
    {
      name: 'Мария С.',
      achievement: 'Открыла свой ателье',
      text: 'Научилась шить с нуля. Теперь шью одежду для всей семьи и открыла небольшое ателье.',
      image: 'linear-gradient(135deg, #C4B5D9 0%, #A8C4A2 100%)',
      rating: 5
    },
    {
      name: 'Елена П.',
      achievement: 'Стала дизайнером одежды',
      text: 'Мечтала научиться шить всю жизнь. Наконец-то нашла время и место. Теперь работаю дизайнером!',
      image: 'linear-gradient(135deg, #A8C4A2 0%, #E8B4B8 100%)',
      rating: 5
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`success-stories-section-sewing ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="success-stories-header-sewing">
          <h2 className="success-stories-title-sewing">Истории успеха</h2>
          <p className="success-stories-description-sewing">
            Наши выпускники достигли невероятных результатов
          </p>
        </div>
        <div className="success-stories-grid-sewing">
          {stories.map((story, index) => (
            <div 
              key={index} 
              className="success-story-card-sewing"
              style={{ animationDelay: `${index * 0.15}s` }}
            >
              <div 
                className="success-story-image-sewing"
                style={{ background: story.image }}
              ></div>
              <div className="success-story-content-sewing">
                <div className="success-story-achievement-sewing">
                  <Award size={20} />
                  <span>{story.achievement}</span>
                </div>
                <p className="success-story-text-sewing">"{story.text}"</p>
                <div className="success-story-author-sewing">
                  <div className="success-story-avatar-sewing">
                    {story.name.split(' ').map(n => n[0]).join('')}
                  </div>
                  <div className="success-story-info-sewing">
                    <div className="success-story-name-sewing">{story.name}</div>
                    <div className="success-story-stars-sewing">
                      {Array.from({ length: 5 }).map((_, i) => (
                        <Star
                          key={i}
                          size={14}
                          fill="#D4AF37"
                          color="#D4AF37"
                        />
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

const BenefitsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const benefits = [
    {
      icon: Award,
      title: 'Опытные мастера',
      description: 'Обучение у профессионалов с многолетним опытом',
      color: 'var(--sewing-rose)'
    },
    {
      icon: Users,
      title: 'Маленькие группы',
      description: 'Индивидуальный подход к каждому студенту',
      color: 'var(--sewing-lavender)'
    },
    {
      icon: Clock,
      title: 'Гибкий график',
      description: 'Выбирайте удобное время для занятий',
      color: 'var(--sewing-sage)'
    },
    {
      icon: Heart,
      title: 'Творческая атмосфера',
      description: 'Вдохновляющая среда для развития ваших талантов',
      color: 'var(--sewing-gold)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`benefits-section-sewing ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="benefits-header-sewing">
          <h2 className="benefits-title-sewing">Преимущества обучения</h2>
          <p className="benefits-description-sewing">
            Почему выбирают наши курсы по шитью
          </p>
        </div>
        <div className="benefits-grid-sewing">
          {benefits.map((benefit, index) => {
            const Icon = benefit.icon
            return (
              <div 
                key={index} 
                className="benefit-card-sewing"
                style={{ 
                  animationDelay: `${index * 0.1}s`,
                  '--benefit-color': benefit.color
                }}
              >
                <div className="benefit-icon-wrapper-sewing">
                  <Icon size={32} />
                </div>
                <h3 className="benefit-title-sewing">{benefit.title}</h3>
                <p className="benefit-text-sewing">{benefit.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default Enroll

